<?php

namespace App\Services;

use App\Models\Setting;
use Exception;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    protected $settings;
    
    public function __construct()
    {
        $tenant = app('tenant');
        $this->settings = Setting::where('tenant_id', $tenant->id)->first();
        
        // Create default settings if none exist
        if (!$this->settings) {
            $this->settings = new Setting([
                'tenant_id' => $tenant->id,
                'currency' => 'BHD',
                'company_name' => 'Your Company Name'
            ]);
        }
    }
    
    /**
     * Format phone number with country code
     */
    public function formatPhoneNumber($phone)
    {
        if (!$this->settings || !$this->settings->whatsapp_auto_format_numbers) {
            return $phone;
        }
        
        // Remove all non-numeric characters except + at the beginning
        $originalPhone = $phone;
        $phone = preg_replace('/[^0-9+]/', '', $phone);
        
        // If phone starts with +, it already has country code
        if (str_starts_with($originalPhone, '+')) {
            return preg_replace('/[^0-9]/', '', $phone); // Remove + for final format
        }
        
        // Define all Asian and Middle Eastern country codes for automatic detection
        $asianCountryCodes = [
            // South Asia
            '880', // Bangladesh
            '91',  // India
            '92',  // Pakistan
            '94',  // Sri Lanka
            '95',  // Myanmar
            '975', // Bhutan
            '977', // Nepal
            '960', // Maldives
            '93',  // Afghanistan
            
            // Southeast Asia
            '60',  // Malaysia
            '62',  // Indonesia  
            '63',  // Philippines
            '65',  // Singapore
            '66',  // Thailand
            '84',  // Vietnam
            '855', // Cambodia
            '856', // Laos
            '673', // Brunei
            '670', // East Timor
            
            // East Asia
            '86',  // China
            '81',  // Japan
            '82',  // South Korea
            '852', // Hong Kong
            '853', // Macau
            '886', // Taiwan
            '976', // Mongolia
            '850', // North Korea
            
            // West Asia / Middle East
            '971', // UAE
            '966', // Saudi Arabia
            '965', // Kuwait
            '974', // Qatar
            '973', // Bahrain
            '968', // Oman
            '967', // Yemen
            '964', // Iraq
            '962', // Jordan
            '961', // Lebanon
            '963', // Syria
            '98',  // Iran
            '90',  // Turkey
            '972', // Israel
            '970', // Palestine
            '357', // Cyprus
            
            // Caucasus
            '994', // Azerbaijan
            '995', // Georgia
            '374', // Armenia
            
            // Central Asia
            '7',   // Kazakhstan/Russia
            '996', // Kyrgyzstan
            '998', // Uzbekistan
            '993', // Turkmenistan
            '992', // Tajikistan
            
            // Other commonly used & Global Coverage
            '1',   // USA/Canada
            '44',  // UK
            '49',  // Germany
            '33',  // France
            '39',  // Italy
            '34',  // Spain
            '61',  // Australia
            '55',  // Brazil
            '27',  // South Africa
            '20',  // Egypt
            
            // Additional European Countries
            '31',  // Netherlands
            '32',  // Belgium
            '41',  // Switzerland
            '43',  // Austria
            '48',  // Poland
            '420', // Czech Republic
            '421', // Slovakia
            '36',  // Hungary
            '40',  // Romania
            '359', // Bulgaria
            '30',  // Greece
            '351', // Portugal
            '46',  // Sweden
            '47',  // Norway
            '45',  // Denmark
            '358', // Finland
            '380', // Ukraine
            '375', // Belarus
            '372', // Estonia
            '371', // Latvia
            '370', // Lithuania
            '353', // Ireland
            '354', // Iceland
            
            // Additional African Countries
            '234', // Nigeria
            '254', // Kenya
            '233', // Ghana
            '255', // Tanzania
            '256', // Uganda
            '212', // Morocco
            '213', // Algeria
            '216', // Tunisia
            '218', // Libya
            '251', // Ethiopia
            
            // Additional American Countries
            '52',  // Mexico
            '54',  // Argentina
            '56',  // Chile
            '57',  // Colombia
            '51',  // Peru
            '58',  // Venezuela
            '593', // Ecuador
            '595', // Paraguay
            '598', // Uruguay
            
            // Additional Oceania Countries
            '64',  // New Zealand
            '679', // Fiji
        ];
        
        // Check if phone already starts with a valid country code
        foreach ($asianCountryCodes as $code) {
            if (str_starts_with($phone, $code)) {
                // Already has country code, return as is
                return $phone;
            }
        }
        
        // Add default country code from settings
        $countryCode = $this->settings->whatsapp_country_code ?? '+880';
        $countryCode = str_replace('+', '', $countryCode);
        
        // Debug logging
        \Log::info('WhatsApp Phone Formatting', [
            'original_phone' => $originalPhone,
            'cleaned_phone' => $phone,
            'default_country_code' => $countryCode,
            'settings_country_code' => $this->settings->whatsapp_country_code ?? 'not set'
        ]);
        
        // Remove leading zero if exists (common in many Asian countries)
        if (str_starts_with($phone, '0')) {
            $phone = substr($phone, 1);
        }
        
        $finalPhone = $countryCode . $phone;
        
        // Final debug log
        \Log::info('WhatsApp Phone Final Result', [
            'final_phone' => $finalPhone
        ]);
        
        return $finalPhone;
    }
    
    /**
     * Replace message variables with actual data
     */
    public function replaceMessageVariables($template, $data = [])
    {
        $defaultData = [
            'company_name' => $this->settings->company_name ?? 'Your Company',
            'customer_name' => $data['customer_name'] ?? 'Valued Customer',
            'customer_phone' => $data['customer_phone'] ?? '',
            'invoice_number' => $data['invoice_number'] ?? 'INV-000',
            'invoice_date' => $data['invoice_date'] ?? date('M d, Y'),
            'due_date' => $data['due_date'] ?? date('M d, Y', strtotime('+30 days')),
            'item_list' => $data['item_list'] ?? 'Items purchased',
            'currency' => $this->settings->currency ?? 'BHD',
            'total_amount' => $data['total_amount'] ?? '0.00',
            'paid_amount' => $data['paid_amount'] ?? '0.00',
            'due_amount' => $data['due_amount'] ?? '0.00',
            'customer_total_due' => $data['customer_total_due'] ?? '0.00',
            'payment_method' => $data['payment_method'] ?? 'Cash',
            'payment_status' => $data['payment_status'] ?? 'Paid',
            'invoice_link' => $data['invoice_link'] ?? url('/'),
        ];
        
        // Merge with provided data
        $data = array_merge($defaultData, $data);
        
        // Build due amount section
        $dueAmountSection = '';
        
        // Show due amount for current sale if exists
        if (isset($data['due_amount']) && floatval($data['due_amount']) > 0) {
            $dueAmountSection .= "⚠️ *Due Amount:* {$data['currency']} {$data['due_amount']}\n";
        }
        
        // Always show customer total outstanding if customer has any due amounts
        if (isset($data['customer_total_due']) && floatval($data['customer_total_due']) > 0) {
            $dueAmountSection .= "📊 *Total Outstanding:* {$data['currency']} {$data['customer_total_due']}\n";
        }
        
        // Build payment summary section - only show paid amount if there's due amount
        $paymentSummarySection = '';
        if (isset($data['due_amount']) && floatval($data['due_amount']) > 0 && 
            isset($data['paid_amount']) && floatval($data['paid_amount']) > 0) {
            $paymentSummarySection = "💳 *Paid Amount:* {$data['currency']} {$data['paid_amount']}\n";
        }
        
        // Add sections to data
        $data['due_amount_section'] = $dueAmountSection;
        $data['payment_summary_section'] = $paymentSummarySection;
        
        // Replace all variables
        foreach ($data as $key => $value) {
            $template = str_replace('{' . $key . '}', $value, $template);
        }
        
        // Add business hours if enabled
        if ($this->settings && $this->settings->whatsapp_include_business_hours && $this->settings->whatsapp_business_hours) {
            $template .= "\n\n📞 *Business Hours:* " . $this->settings->whatsapp_business_hours;
        }
        
        return $template;
    }
    
    /**
     * Generate WhatsApp URL for QR code or direct link
     */
    public function generateWhatsAppUrl($phone, $message)
    {
        // Format phone number
        $formattedPhone = $this->formatPhoneNumber($phone);
        
        // Encode message for URL
        $encodedMessage = urlencode($message);
        
        // Generate WhatsApp URL using official WhatsApp API
        return "https://api.whatsapp.com/send/?phone={$formattedPhone}&text={$encodedMessage}";
    }
    
    /**
     * Generate QR code URL using Google Charts API
     */
    public function generateQrCodeUrl($phone, $message, $size = 300)
    {
        $whatsappUrl = $this->generateWhatsAppUrl($phone, $message);
        $encodedUrl = urlencode($whatsappUrl);
        
        return "https://chart.googleapis.com/chart?chs={$size}x{$size}&cht=qr&chl={$encodedUrl}&choe=UTF-8";
    }
    
    /**
     * Generate QR code using QR Server API (alternative)
     */
    public function generateQrCodeUrlAlt($phone, $message, $size = 300)
    {
        $whatsappUrl = $this->generateWhatsAppUrl($phone, $message);
        $encodedUrl = urlencode($whatsappUrl);
        
        return "https://api.qrserver.com/v1/create-qr-code/?size={$size}x{$size}&data={$encodedUrl}";
    }
    
    /**
     * Create WhatsApp data for invoice
     */
    public function createInvoiceWhatsApp($phone, $invoiceData)
    {
        // Always use the updated template for better formatting
        $template = $this->getDefaultInvoiceTemplate();
        $message = $this->replaceMessageVariables($template, $invoiceData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Create WhatsApp data for welcome message
     */
    public function createWelcomeWhatsApp($phone, $customerData)
    {
        $template = $this->settings->whatsapp_welcome_message ?? $this->getDefaultWelcomeTemplate();
        $message = $this->replaceMessageVariables($template, $customerData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Create WhatsApp data for payment reminder
     */
    public function createPaymentReminderWhatsApp($phone, $invoiceData)
    {
        $template = $this->settings->whatsapp_payment_reminder_message ?? $this->getDefaultReminderTemplate();
        $message = $this->replaceMessageVariables($template, $invoiceData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Create WhatsApp data for delivery notification
     */
    public function createDeliveryWhatsApp($phone, $orderData)
    {
        $template = $this->settings->whatsapp_delivery_message ?? $this->getDefaultDeliveryTemplate();
        $message = $this->replaceMessageVariables($template, $orderData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Create WhatsApp data for thank you message
     */
    public function createThankYouWhatsApp($phone, $purchaseData)
    {
        $template = $this->settings->whatsapp_thank_you_message ?? $this->getDefaultThankYouTemplate();
        $message = $this->replaceMessageVariables($template, $purchaseData);
        
        return [
            'phone' => $this->formatPhoneNumber($phone),
            'message' => $message,
            'whatsapp_url' => $this->generateWhatsAppUrl($phone, $message),
            'qr_code_url' => $this->generateQrCodeUrl($phone, $message),
            'qr_code_url_alt' => $this->generateQrCodeUrlAlt($phone, $message)
        ];
    }
    
    /**
     * Default templates
     */
    private function getDefaultInvoiceTemplate()
    {
        return "🧾 *Invoice #{invoice_number}*\n\n" .
               "👤 *Customer:* {customer_name}\n" .
               "📞 *Phone:* {customer_phone}\n" .
               "📅 *Date:* {invoice_date}\n\n" .
               "📦 *Items:*\n{item_list}\n\n" .
               "💰 *Total Amount:* {currency} {total_amount}\n" .
               "💳 *Payment Method:* {payment_method}\n" .
               "📊 *Payment Status:* {payment_status}\n" .
               "{payment_summary_section}" .
               "{due_amount_section}" .
               "🙏 *Thank you for your business!*\n\n" .
               "🔗 *View Invoice:* {invoice_link}";
    }
    
    private function getDefaultWelcomeTemplate()
    {
        return "🎉 *Welcome to {company_name}!*\n\n" .
               "Hello {customer_name}! 👋\n\n" .
               "Thank you for choosing us for your mobile phone needs.\n\n" .
               "📱 For any assistance, feel free to contact us.\n\n" .
               "Best regards,\n{company_name} Team 🙏";
    }
    
    private function getDefaultReminderTemplate()
    {
        return "💳 *Payment Reminder*\n\n" .
               "Dear {customer_name},\n\n" .
               "This is a friendly reminder that your invoice #{invoice_number} for {currency} {total_amount} is due.\n\n" .
               "📅 *Due Date:* {due_date}\n\n" .
               "Please make the payment at your earliest convenience.\n\n" .
               "🔗 *View Invoice:* {invoice_link}\n\n" .
               "Thank you! 🙏";
    }
    
    private function getDefaultDeliveryTemplate()
    {
        return "🚚 *Order Delivery Update*\n\n" .
               "Dear {customer_name},\n\n" .
               "Your order #{invoice_number} is ready for delivery! 📦\n\n" .
               "📱 *Items:*\n{item_list}\n\n" .
               "Our delivery team will contact you shortly.\n\n" .
               "Thank you! 🙏";
    }
    
    private function getDefaultThankYouTemplate()
    {
        return "🎉 *Thank You for Your Purchase!*\n\n" .
               "Dear {customer_name},\n\n" .
               "Thank you for choosing {company_name}! ✅\n\n" .
               "🧾 *Invoice:* #{invoice_number}\n" .
               "💰 *Amount:* {currency} {total_amount}\n\n" .
               "We hope you enjoy your new device! 📱\n\n" .
               "Best regards,\n{company_name} Team 🙏";
    }
}